local super = require "View"

Label = super:new()

Label.titleStyle = 'title'
Label.labelStyle = 'label'

local PAINTS = {
    [Label.titleStyle] = ColorScheme.titlePaint,
    [Label.labelStyle] = ColorScheme.labelPaint,
}

local FONTS = {
    [Label.titleStyle] = TypographyScheme.titleFont,
    [Label.labelStyle] = TypographyScheme.blockFont,
}

local defaults = {
    text = "Text",
    style = Label.labelStyle,
    halign = 0,
    valign = 1,
}

local nilDefaults = {
}

local inspectorInfo = {
    {'Alignment', {'halign'}, nil, 'Alignment'},
}

function Label:new()
    self = super.new(self)
    
    for k, v in pairs(defaults) do
        self:addProperty(k, v)
    end
    for _, k in pairs(nilDefaults) do
        self:addProperty(k)
    end
    
    return self
end

function Label:unarchivePaint(archived)
    -- NOTE: Version 1.3.2 and earlier stored label text colors separately from the color scheme.
    self._legacyPaint = unarchive(archived)
end

function Label:unarchiveFont(archived)
    -- NOTE: Version 1.4.2 and earlier stored label fonts without a typography scheme.
    self._legacyFont = unarchive(archived)
end

function Label:unarchived()
    local style = self:getProperty('style')
    if self._legacyPaint then
        self:setPaint(PAINTS[style], self._legacyPaint)
    end
    if self._legacyFont then
        self:setFont(FONTS[style], self._legacyFont)
    end
    super.unarchived(self)
end

function Label:getInspectors()
    local list = super.getInspectors(self)
    local style = self:getProperty('style')
    list:add(self:createColorInspector(PAINTS[style], nil, 'Color'))
    list:add(self:createFontInspector(FONTS[style], nil, 'Font'))
    for i = 1, #inspectorInfo do
        local info = inspectorInfo[i]
        list:add(self:createInspector(unpack(info)))
    end
    return list
end

function Label:draw(canvas)
    local rect = self:rect()
    if canvas:isHitTest() then
        canvas:fill(Path.rect(rect))
        return
    end
    local text = self:getProperty('text')
    local style = self:getProperty('style')
    local paint = self:getPaint(PAINTS[style])
    local font = self:getFont(FONTS[style])
    local halign = self:getProperty('halign')
    local valign = self:getProperty('valign')
    canvas:preserve(function(canvas)
        canvas:clipRect(rect:insetXY(-1, -1))
        WrappedTextStamp(canvas, rect, text, paint, font, halign, valign)
    end)
end

function Label:getEditableComponent(x, y)
    local style = self:getProperty('style')
    local hooks = {
        text = 'text',
        halign = 'halign',
        valign = 'valign',
    }
    local inspector = self:createInspector('string', hooks, 'Text')
    inspector:addHook(self:getPaintHook(PAINTS[style]), 'paint')
    inspector:addHook(self:getFontHook(FONTS[style]), 'font')
    local hook
    hook = Hook:new(
        function()
            return self:rect()
        end,
        function(value) end)
    inspector:addHook(hook, 'rect')
    inspector:addHook(Hook:new(true), 'multiline')
    return inspector
end

return Label
